package com.artstore.session;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import javax.servlet.http.HttpSession;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.artstore.Core;
import com.artstore.account.Account;
import com.artstore.entity.IEntity;
import com.artstore.managers.AbstractManager;
import com.artstore.managers.ISessionManager;

@Component
public class SessionManager<TID> extends AbstractManager implements ISessionManager<TID>
{
	/*
	 * HttpSession stores: UUID 'sessionID' ; session ID 'isLogged' ; true when
	 * entity is logged (linking is valid); false otherwise
	 * 
	 * 
	 * HttpSession is unlinked when it's not linked HttpSession is linked when it
	 * contains 'sessionID' attribute and linkage is valid HttpSession's Linkage is
	 * valid when Session is valid
	 * 
	 * Linkage cannot be overridden, only removed
	 */

	protected static class SessionData<TID>
	{
		IEntity<TID>  linkedEntity;

		LocalDateTime initDate;

		public SessionData(IEntity<TID> linkedEntity)
		{
			this.linkedEntity = linkedEntity;
			this.initDate = LocalDateTime.now();
		}

		public IEntity<TID> getLinkedEntity()
		{
			return linkedEntity;
		}

		public LocalDateTime getInitDate()
		{
			return initDate;
		}

		public void Refresh()
		{
			initDate = LocalDateTime.now();
		}
	}

	/* <HttpSession.sessionID, Session Data> */
	private final Map<TID, SessionData<TID>> sessionMap = new HashMap<>();
	private IIDGenerator<TID> idGenerator;
	
	protected Duration                         sessionExpirationPeriod;
	
	@Autowired
	public SessionManager(Core core)
	{
		this(core, null);
	}
	
	public SessionManager(Core core, IIDGenerator<TID> idGenerator)
	{
		super(core);
		
		this.idGenerator = idGenerator;

		sessionExpirationPeriod = Duration.ofMinutes(15);
	}
	
	public static SessionManager<UUID> Create(Core core)
	{
		return new SessionManager<>(core, new IIDGenerator<UUID>()
				{
					public UUID Generate()
					{
						return UUID.randomUUID();
					}
				});
	}
	
	@SuppressWarnings("unchecked")
	TID GetSessionID(HttpSession session)
	{
		return (TID) session.getAttribute("sessionID");
	}

	SessionData<TID> GetSessionData(HttpSession session)
	{
		return sessionMap.get(GetSessionID(session));
	}

	boolean IsUnlinked(HttpSession session)
	{
		return !IsLinked(session);
	}

	boolean IsLinked(HttpSession session)
	{
		TID sessionID = GetSessionID(session);
		if (sessionID == null)
		{
			return false;
		}

		return hasGoodTiming(sessionID);
	}

	private boolean hasGoodTiming(TID sessionID)
	{
		SessionData<TID> sessionData = sessionMap.get(sessionID);

		return sessionData == null ? false
				: (Duration.between(sessionData.getInitDate(), LocalDateTime.now())
						.compareTo(sessionExpirationPeriod) < 0);
	}

	@Override
	public boolean ValidateSession(HttpSession session)
	{
		return IsLinked(session);
	}

	@Override
	public TID GetEntityID(HttpSession session)
	{
		TID sessionID = GetSessionID(session);
		if (sessionID == null)
		{
			return null;
		}

		SessionData<TID> sessionData = sessionMap.get(sessionID);

		return sessionData == null ? null : sessionData.getLinkedEntity().GetID();
	}

	@Override
	public IEntity<TID> GetEntity(HttpSession session)
	{
		TID sessionID = GetSessionID(session);
		if (sessionID == null)
		{
			return null;
		}

		SessionData<TID> sessionData = sessionMap.get(sessionID);

		return sessionData == null ? null : sessionData.getLinkedEntity();
	}
	
	public Account GetAccount(HttpSession session)
	{
		TID sessionID = GetSessionID(session);
		if (sessionID == null || !(sessionID instanceof UUID))
		{
			return null;
		}

		@SuppressWarnings("unchecked")
		SessionData<UUID> sessionData = (SessionData<UUID>)sessionMap.get(sessionID);

		return sessionData == null || !(sessionData.getLinkedEntity() instanceof Account) ? null : Core().GetAccountManager().GetAccount(sessionData.getLinkedEntity().GetID());
	}

	@Override
	public TID LinkSession(HttpSession session, IEntity<TID> entity)
	{
		if (IsLinked(session))
		{
			return null;
		}

		TID sessionID = idGenerator.Generate();
		SessionData<TID> sessionData = new SessionData<TID>(entity);

		sessionMap.put(sessionID, sessionData);
		session.setAttribute("sessionID", sessionID);
		session.setAttribute("ent", entity);
		session.setAttribute("isLogged", "true");
		
		return sessionID;
	}

	@Override
	public boolean UnlinkSession(HttpSession session)
	{
		TID sessionID = GetSessionID(session);
		if (sessionID == null)
		{
			return false;
		}

		sessionMap.remove(sessionID);
		session.removeAttribute("sessionID");
		session.removeAttribute("ent");
		session.setAttribute("isLogged", "false");

		return true;
	}

	@Override
	public boolean RefreshSessionLinkage(HttpSession session)
	{
		if (IsUnlinked(session))
		{
			return false;
		}

		GetSessionData(session).Refresh();

		return true;
	}

}
