package com.artstore.economy;

import javax.transaction.Transactional;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.artstore.Core;
import com.artstore.managers.AbstractManager;
import com.artstore.managers.IEconomyManager;
import com.artstore.repositories.CurrencyRepository;
import com.artstore.repositories.EconomyRepository;

@Component
public class EconomyManager extends AbstractManager implements IEconomyManager
{	
	private EconomyRepository economyRepo;
	private CurrencyRepository currencyRepo;

	@Autowired
	public EconomyManager(Core core, EconomyRepository economyRepo, CurrencyRepository currencyRepo)
	{
		super(core);
		
		this.economyRepo = economyRepo;
		this.currencyRepo = currencyRepo;
	}

	@Override
	public boolean Initialize()
	{
		return super.Initialize();
	}

	@Override
	public boolean Dispose()
	{
		return super.Dispose();
	}

	@Override
	public EconomyRepository GetRepo()
	{
		return economyRepo;
	}
	
	public CurrencyRepository GetCurrencyRepo()
	{
		return currencyRepo;
	}
	
	public CompositeMoney CalcBalance(EconomyEntity entity)
	{
		CompositeMoney res = new CompositeMoney();
		
		for(Transfer transfer : entity.getTransfers())
		{
			if(transfer.getSource().equals(entity))
			{
				res.Subtract(transfer.getMonetaryValue());
			}
			else if(transfer.getDestination().equals(entity))
			{
				res.Add(transfer.getMonetaryValue());
			}
		}
		
		return res;
	}

	public boolean IsPossible(Transfer transfer)
	{
		return transfer.getSource().getQuantities().IsGreaterThan(transfer.getMonetaryValue());
	}

	@Override
	public boolean ApplyTransferRequest(TransferRequest request)
	{
		if (request.isApproved())
		{
			return false;
		}

		EconomyEntity entity = economyRepo.findByTransferCode(request.gettransferCode());
		if (entity == null)
		{
			return false;
		}

		if (PostTransfer(entity, request.getIncomingMoney()))
		{
			request.setApproved();

			return true;
		}

		return false;
	}

	@Transactional
	public boolean PostTransfer(Transfer transfer)
	{
		if (IsPossible(transfer))
		{
			return InsertTransfer(transfer);
		}

		return false;
	}

	@Transactional
	protected boolean InsertTransfer(Transfer transfer)
	{
		if (transfer.getSource() != null)
		{
			transfer.getSource().getQuantities().Subtract(transfer.getMonetaryValue());
			transfer.getSource().getTransfers().add(transfer);
		}
		
		if (transfer.getDestination() != null)
		{
			transfer.getDestination().getQuantities().Add(transfer.getMonetaryValue());
			transfer.getDestination().getTransfers().add(transfer);
		}

		return true;
	}
}
