package com.artstore.controllers;

import java.util.Locale;
import java.util.Map;

import org.apache.commons.validator.routines.EmailValidator;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.configurationprocessor.json.JSONException;
import org.springframework.boot.configurationprocessor.json.JSONObject;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

import com.artstore.Core;
import com.artstore.Utils;
import com.artstore.account.Account;
import com.artstore.account.Code;
import com.artstore.managers.IRegistrationManager.RegResult;

@Controller
public class RegistrationController extends AbstractController
{
	@Autowired
	public RegistrationController(Core core)
	{
		super(core);
	}

	public static enum RegValidationResponse
	{
		OK, USER_MIN, USER_MAX, PSSWD_MIN, PSSWD_MAX, USER_PAT, PSSWD_PAT, MAIL_PAT, PSSWD_NMATCH, USER_OCCUPIED,
		MAIL_OCCUPIED, ERROR
	}

	private int USER_MINLENGTH  = 6;
	private int USER_MAXLENGTH  = 12;

	private int PSSWD_MINLENGTH = 6;
	private int PSSWD_MAXLENGTH = 12;

	public boolean Validate(JSONObject res, String username, String psswd, String psswd2, String mail)
	{
		try
		{
			if (username == null || username.length() < USER_MINLENGTH)
			{
				res.put("resp", RegValidationResponse.USER_MIN);
				res.put("val", USER_MINLENGTH);
			}
			else if (USER_MAXLENGTH < username.length())
			{
				res.put("resp", RegValidationResponse.USER_MAX);
				res.put("val", USER_MAXLENGTH);
			}
			else if (psswd == null || psswd.length() < PSSWD_MINLENGTH)
			{
				res.put("resp", RegValidationResponse.PSSWD_MIN);
				res.put("val", PSSWD_MINLENGTH);
			}
			else if (PSSWD_MAXLENGTH < psswd.length())
			{
				res.put("resp", RegValidationResponse.PSSWD_MAX);
				res.put("val", PSSWD_MAXLENGTH);
			}
			else if (!username.matches("[a-zA-Z0-9]*"))
			{
				res.put("resp", RegValidationResponse.USER_PAT);
			}
			else if (!psswd.matches("[a-zA-Z0-9]*"))
			{
				res.put("resp", RegValidationResponse.PSSWD_PAT);
			}
			else if (!EmailValidator.getInstance().isValid(mail))
			{
				res.put("resp", RegValidationResponse.MAIL_PAT);
			}
			else if (!psswd.equals(psswd2))
			{
				res.put("resp", RegValidationResponse.PSSWD_NMATCH);
			}
			else if (Core().GetAccountManager().GetAccount(username) != null)
			{
				res.put("resp", RegValidationResponse.USER_OCCUPIED);
			}
			else if (Core().GetAccountManager().GetAccountByMail(mail) != null)
			{
				res.put("resp", RegValidationResponse.MAIL_OCCUPIED);
			}
			else
			{
				res.put("resp", RegValidationResponse.OK);

				return true;
			}
		}
		catch(JSONException ex)
		{
			ex.printStackTrace();
		}

		return false;
	}

	@RequestMapping(value = "/registration/validate", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_VALUE)
	@ResponseBody
	public String validate(@RequestParam Map<String, String> params)
	{
		String username = params.get("username");
		String psswd = params.get("password");
		String psswd2 = params.get("password2");
		String mail = params.get("mail").toLowerCase();

		JSONObject res = new JSONObject();
		Validate(res, username, psswd, psswd2, mail);

		return res.toString();
	}

	@RequestMapping(value = "/registration/requestRegistration", method = RequestMethod.POST, produces = MediaType.TEXT_PLAIN_VALUE)
	@ResponseBody
	public String register(@RequestParam("username") String username, @RequestParam("password") String password,
			@RequestParam("mail") String mail, Model model, RedirectAttributes ra)
	{
		RegResult res = Core().GetRegistrationManager().RegisterAccount(username, password, mail);

		return res.name();
	}

	@RequestMapping(value = "/registration/success", method = RequestMethod.POST)
	public String success()
	{
		return "/index";
	}

	@RequestMapping(value = "/registration/activate", method = RequestMethod.GET)
	public String activate(@RequestParam("account") String account, @RequestParam("code") String code, Locale locale,
			Model model, RedirectAttributes ra)
	{
		Account acc = Core().GetAccountManager().GetAccount(account);
		if (acc == null)
		{
			return Utils.Info("Błąd podczas aktywacji konta.", model, ra);
		}

		if (acc.isActivated())
		{
			return Utils.Info("Konto już jest aktywne!", model, ra);
		}

		return Utils.Info(Core().GetRegistrationManager().ActivateAccount(acc, code) ? "Konto zostało aktywowane!"
				: "Błąd podczas aktywacji konta (błędny kod?)!", model, ra);
	}

	@RequestMapping(value = "/registration/sendActivationMail", method = RequestMethod.GET, produces= MediaType.TEXT_PLAIN_VALUE)
	@ResponseBody
	public String sendActivationMail(@RequestParam("username") String username, Locale locale, Model model,
			RedirectAttributes ra)
	{
		Account acc = Core().GetAccountManager().GetAccount(username);

		if (acc == null || acc.isActivated()) { return "null"; }

		return Core().GetCodeManager().SendCodeMail(Code.Activation, acc) ? "true" : "false";
	}
}
