package com.artstore.config;

import java.io.IOException;
import java.util.Properties;
import java.util.UUID;

import javax.persistence.EntityManagerFactory;
import javax.sql.DataSource;

import org.springframework.boot.autoconfigure.domain.EntityScan;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.PropertySource;
import org.springframework.context.annotation.PropertySources;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.support.PropertiesLoaderUtils;
import org.springframework.dao.annotation.PersistenceExceptionTranslationPostProcessor;
import org.springframework.data.jpa.repository.config.EnableJpaRepositories;
import org.springframework.jdbc.datasource.DriverManagerDataSource;
import org.springframework.orm.jpa.JpaTransactionManager;
import org.springframework.orm.jpa.JpaVendorAdapter;
import org.springframework.orm.jpa.LocalContainerEntityManagerFactoryBean;
import org.springframework.orm.jpa.vendor.HibernateJpaVendorAdapter;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.annotation.EnableTransactionManagement;

import com.artstore.Core;
import com.artstore.session.SessionManager;

@ComponentScan(basePackages={"com.artstore"})
@Configuration
@ConfigurationProperties
@EnableJpaRepositories({"com.artstore"})
@EnableTransactionManagement
@EntityScan(basePackages = {"com.artstore"} )
@PropertySources({ @PropertySource("classpath:application.properties"), @PropertySource("file:server.properties") })
public class PersistenceContext
{
	private static final Properties props = ExternalConfigUtils.GetProperties("server");

	@Bean
	public LocalContainerEntityManagerFactoryBean entityManagerFactory()
	{
		LocalContainerEntityManagerFactoryBean em = new LocalContainerEntityManagerFactoryBean();
		em.setJpaProperties(additionalProperties());
		em.setDataSource(dataSource());
		em.setPackagesToScan(new String[] { "com.artstore" });

		JpaVendorAdapter vendorAdapter = new HibernateJpaVendorAdapter();
		em.setJpaVendorAdapter(vendorAdapter);
		em.setJpaProperties(additionalProperties());

		return em;
	}
	
	@Bean(name="defaultSessionManager")
	public SessionManager<UUID> sessionManager(Core core)
	{
		return SessionManager.Create(core);
	}

	@Bean
	public DataSource dataSource()
	{
		DriverManagerDataSource dataSource = new DriverManagerDataSource();
		dataSource.setDriverClassName(props.getProperty("database.driver"));
		dataSource.setUrl(props.getProperty("database.url"));
		dataSource.setUsername(props.getProperty("database.username"));

		if (props.containsKey("database.password"))
		{
			dataSource.setPassword(props.getProperty("database.password"));
		}

		return dataSource;
	}

	@Bean
	public PlatformTransactionManager transactionManager(EntityManagerFactory emf)
	{
		JpaTransactionManager transactionManager = new JpaTransactionManager();
		transactionManager.setEntityManagerFactory(emf);

		return transactionManager;
	}

	@Bean
	public PersistenceExceptionTranslationPostProcessor exceptionTranslation()
	{
		return new PersistenceExceptionTranslationPostProcessor();
	}

	Properties additionalProperties()
	{
		Properties properties;

		try
		{
			properties = PropertiesLoaderUtils.loadProperties(new ClassPathResource("jpa.properties"));
		}
		catch(IOException ex)
		{
			ex.printStackTrace();

			properties = new Properties();
		}

		ExternalConfigUtils.LoadProperties("server", properties, "database");

		return properties;
	}
}