package com.artstore.config;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.util.Collections;
import java.util.Enumeration;
import java.util.Properties;
import java.util.TreeSet;

import org.apache.commons.configuration.CombinedConfiguration;
import org.apache.commons.configuration.ConfigurationConverter;
import org.apache.commons.configuration.PropertiesConfiguration;
import org.apache.commons.configuration.PropertiesConfigurationLayout;
import org.apache.commons.configuration.tree.OverrideCombiner;
import org.springframework.core.io.ClassPathResource;

public class ExternalConfigUtils
{
	public static Properties PrepareProperties(String filename)
	{
		return PrepareProperties(filename, null);
	}

	public static Properties PrepareProperties(String filename, String jarDefaultFileName)
	{
		InputStreamReader reader = null;
		OutputStreamWriter writer = null;
		
		PropertiesConfiguration defaultConfig = null;
		PropertiesConfiguration currentConfig = new PropertiesConfiguration();

		try
		{
			/* Loading of default properties from resource, if present */
			if (jarDefaultFileName != null)
			{
				reader = new InputStreamReader(new ClassPathResource(jarDefaultFileName).getInputStream());
				
				defaultConfig = new PropertiesConfiguration();
				defaultConfig.load(reader);
				
				reader.close();
			}

			/*
			 * Loading of recent properties from config file (overriding default values with
			 * the set ones)
			 */
			File file = new File(filename);
			file.createNewFile();
			reader = new InputStreamReader(new FileInputStream(file));
			currentConfig.load(reader);
			reader.close();
			
			CombinedConfiguration props = new CombinedConfiguration();
			props.setNodeCombiner(new OverrideCombiner());
			props.addConfiguration(currentConfig);
			props.addConfiguration(defaultConfig);

			/* Saving/updating config file */			
			PropertiesConfiguration finalFile = new PropertiesConfiguration();   
			finalFile.append(props);
			PropertiesConfigurationLayout layout = new PropertiesConfigurationLayout(finalFile, defaultConfig.getLayout()); // Copying the layout from origin file
			
			writer = new OutputStreamWriter(new FileOutputStream(file));
			layout.save(writer);
			writer.close();

			return ConfigurationConverter.getProperties(currentConfig);
		}
		catch(Exception ex)
		{
			ex.printStackTrace();
		}
		finally
		{
			try
			{
				if (reader != null)
				{
					reader.close();
				}
				if (writer != null)
				{
					writer.close();
				}
			}
			catch(IOException ex)
			{
				ex.printStackTrace();
			}
		}

		return null;
	}

	public static void PushProperties(Properties from, Properties to, String prefix)
	{
		for(String prop:from.stringPropertyNames())
		{
			if (!to.containsKey(prop))
			{
				to.setProperty(prop, from.getProperty(prop));
			}
		}
	}

	public static boolean LoadProperties(String propTitle, Properties propsTo, String... prefixes)
	{
		return LoadProperties(propTitle + ".properties", "config/" + propTitle + ".properties", propsTo, prefixes);
	}

	public static boolean LoadProperties(String filename, String jarDefaultFileName, Properties propsTo,
			String... prefixes)
	{
		Properties propsFrom = PrepareProperties(filename, jarDefaultFileName);
		if (propsFrom == null)
		{
			return false;
		}

		for(String p:prefixes)
		{
			PushProperties(propsFrom, propsTo, p + ".");
		}

		return true;
	}

	public static Properties GetProperties(String title)
	{
		@SuppressWarnings("serial")
		Properties res = new Properties(){

			@Override
		    public synchronized Enumeration<Object> keys() {
		        return Collections.enumeration(new TreeSet<Object>(super.keySet()));
		    }
		};

		Properties cfg = PrepareProperties(title + ".properties", "config/" + title + ".properties");
		PushProperties(cfg, res, title + ".");

		return res;
	}

	public static boolean UpdateProperty(String filename, String propKey, String propValue)
	{
		InputStreamReader reader = null;
		OutputStreamWriter writer = null;
		
		PropertiesConfiguration config = new PropertiesConfiguration();
		PropertiesConfigurationLayout layout = new PropertiesConfigurationLayout(config);

		try
		{
			/*
			 * Loading of recent properties from config file (overriding default values with
			 * the set ones)
			 */
			File file = new File(filename);
			file.createNewFile();
			reader = new InputStreamReader(new FileInputStream(file));
			layout.load(reader);
			reader.close();

			/* Applying change */
			config.setProperty(propKey, propValue);

			/* Saving/updating config file */
			writer = new OutputStreamWriter(new FileOutputStream(file));
			layout.save(writer);
			writer.close();
			
			return true;
		}
		catch(Exception ex)
		{
			ex.printStackTrace();
		}
		finally
		{
			try
			{
				if (reader != null)
				{
					reader.close();
				}
				if (writer != null)
				{
					writer.close();
				}
			}
			catch(IOException ex)
			{
				ex.printStackTrace();
			}
		}

		return false;
	}
	
	public static void UpdateInit()
	{
		UpdateProperty("server.properties", "server.init", "false");
	}
}
