package com.artstore.account;

import java.nio.charset.StandardCharsets;

import javax.transaction.Transactional;
import javax.transaction.Transactional.TxType;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.artstore.Core;
import com.artstore.managers.AbstractManager;
import com.artstore.managers.IAccountManager;
import com.artstore.managers.IRegistrationManager;

@Component
public class RegistrationManager extends AbstractManager implements IRegistrationManager
{
	@Autowired
	public RegistrationManager(Core core)
	{
		super(core);
	}
	
	@Transactional(value = TxType.REQUIRES_NEW)
	public RegResult RegisterAccount(String username, String password, String email)
	{
		IAccountManager aM = Core().GetAccountManager();

		if (aM.GetAccount(username) != null)
		{
			return RegResult.ALREADY_EXISTS;
		}

		byte[] salt = SecurityUtils.GenerateSalt();
		byte[] passwd = SecurityUtils.Hash(password.getBytes(StandardCharsets.UTF_8), salt);

		Account newAcc = new Account();
		newAcc.setUsername(username);
		newAcc.setPassword(passwd);
		newAcc.setPasswordSalt(salt);
		newAcc.setMailbox(email);

		if(aM.Create(newAcc) == null) { return RegResult.DB_ERROR; }
		
		Core().GetCodeManager().SendCodeMail(Code.Activation, newAcc);
		return RegResult.OK;
	}

	public boolean ActivateAccount(Account account, String code)
	{
		if (Core().GetCodeManager().Verify(Code.Activation, account, code))
		{
			Activate(account);

			return true;
		}

		return false;
	}
	
	public boolean RecoveryAccount(Account account, String code)
	{
		if (Core().GetCodeManager().Verify(Code.Recovery, account, code))
		{
			String psswd = Core().GetAccountManager().SetNewPassword(account);			
			if(psswd == null) { return false; }
			
			Core().GetMailManager().SendMail("art.store.art.store@gmail.com", account.getMailbox(), "Art Store Nowe Hasło",
					"Nowe hasło: " + psswd);
			
			return true;
		}

		return false;
	}

	@Transactional
	private void Activate(Account acc)
	{
		acc.setActivated(true);
		
		Core().GetAccountManager().Update(acc);
	}
}
