package com.artstore.account;

import java.util.List;
import java.util.UUID;

import javax.servlet.http.HttpSession;
import javax.transaction.Transactional;

import org.apache.commons.lang3.RandomStringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import com.artstore.Core;
import com.artstore.managers.AbstractManager;
import com.artstore.managers.IAccountManager;
import com.artstore.managers.ISessionManager;
import com.artstore.permissions.IPermissible;
import com.artstore.permissions.IPermissibleGroup;
import com.artstore.permissions.PermissionManager;
import com.artstore.permissions.permissions.RootPermission;
import com.artstore.repositories.AccountRepository;

@Component
public class AccountManager extends AbstractManager implements IAccountManager
{
	private AccountRepository repository;
	@Autowired
	private PermissionManager pM;

	private boolean           isRootEnabled;

	@Autowired
	public AccountManager(Core core, AccountRepository repository)
	{
		super(core);

		this.repository = repository;
	}

	@Override
	public boolean Initialize()
	{
		isRootEnabled = Boolean.parseBoolean(Core().GetProperty("server.root.enabled"));

		if (Core().IsInitRequested())
		{
			if (GetAccount("root") == null)
			{
				String rootLogin = Core().GetProperty("server.root.login");
				String rootPassword = Core().GetProperty("server.root.password");

				Account rootAccount = Account.Create(rootLogin, rootPassword, "");
				rootAccount.setActivated(Boolean.parseBoolean(Core().GetProperty("server.root.enabled")));
				rootAccount.Grant(new RootPermission(Core()));
				rootAccount.setAccountType(3);

				IPermissibleGroup<UUID> group = pM.GetGroup("Administrator");

				if (group != null)
				{
					List<IPermissible<UUID>> list = rootAccount.GetSupers();
					list.add(group);
				}

				Create(rootAccount);
			}
		}

		return true;
	}

	@Override
	public boolean Dispose()
	{
		return true;
	}

	public AccountRepository GetRepo()
	{
		return repository;
	}

	@Override
	public boolean IsRootSupported()
	{
		return isRootEnabled;
	}

	public Iterable<Account> FetchAll()
	{
		return GetRepo().findAll();
	}

	public Page<InfoAccount> Fetch(int pageNumber, int pageSize)
	{
		Pageable pageableRequest = PageRequest.of(pageNumber, pageSize);

		@SuppressWarnings("unchecked")
		Page<InfoAccount> users = (Page<InfoAccount>) GetRepo().findAllProjectedBy(pageableRequest);

		return users;
	}

	@Override
	public Account GetAccount(UUID id)
	{
		return GetRepo().findById(id).get();
	}

	@Override
	public Account GetAccount(String username)
	{
		return GetRepo().findByUsername(username);
	}

	@Override
	public Account GetAccountByMail(String mail)
	{
		return GetRepo().findByMailbox(mail);
	}

	@Override
	public boolean VerifyCredentials(String username, String password)
	{
		Account account = GetAccount(username);
		if (account == null)
		{
			return false;
		}

		return account.isActivated()
				&& SecurityUtils.ValidatePassword(password, account.getPasswordSalt(), account.getPassword());
	}

	public String SetNewPassword(Account acc)
	{
		return SetNewPassword(acc, RandomStringUtils.randomAlphanumeric(12));
	}

	@Transactional
	public String SetNewPassword(Account acc, String psswd)
	{
		if (acc == null)
		{
			return null;
		}

		byte[] salt = SecurityUtils.GenerateSalt();
		byte[] hash = SecurityUtils.Hash(psswd, salt);

		acc.setPassword(hash);
		acc.setPasswordSalt(salt);

		return Update(acc) == null ? null : psswd;
	}

	@Transactional
	public boolean SetNewMail(Account acc, String mail)
	{
		if (acc == null)
		{
			return false;
		}

		acc.setMailbox(mail);

		return Update(acc) != null;
	}

	@Transactional
	public boolean SetPersonalData(Account acc, String name, String phoneNumber, String contactMail, String address)
	{
		if (acc == null)
		{
			return false;
		}

		acc.setName(name);
		acc.setPhoneNumber(phoneNumber);
		acc.setContactMailbox(contactMail);
		acc.setAddress(address);

		return Update(acc) != null;
	}

	@Override
	public UUID Login(String username, String password, HttpSession session)
	{
		ISessionManager<UUID> sessionManager = Core().GetSessionManager();

		if (!sessionManager.ValidateSession(session)
				&& Core().GetAccountManager().VerifyCredentials(username, password))
		{
			Account account = GetAccount(username);

			return sessionManager.LinkSession(session, account);
		}

		return null;
	}
}
