package com.artstore;

import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Component;

import com.artstore.account.CodeManager;
import com.artstore.config.ExternalConfigUtils;
import com.artstore.entity.IEntity;
import com.artstore.log.LogEntry;
import com.artstore.log.LogLevel;
import com.artstore.managers.IAccountManager;
import com.artstore.managers.IEconomyManager;
import com.artstore.managers.IEventManager;
import com.artstore.managers.IItemManager;
import com.artstore.managers.ILogManager;
import com.artstore.managers.IMailManager;
import com.artstore.managers.IManager;
import com.artstore.managers.IPaymentManager;
import com.artstore.managers.IPermissionManager;
import com.artstore.managers.IRegistrationManager;
import com.artstore.managers.ISessionManager;

@Component
public class Core
{
	@Value("${version}")
	private String                             version;

	public String lol = "@project.version@";
	
	@Value("${server.address}")
	private String                             serverAddress;

	@Value("${server.port}")
	private short                              serverPort;

	@Value("${server.wwwLink}")
	private String                             serverWWWLink;

	private DbCred                             dbCred      = new DbCred();

	private OutputStream                       stdout;

	private ConfigurableApplicationContext context;
	@Autowired
	private EntityManagerFactory               entityManagerFactory;

	@Autowired
	private IAccountManager                    accountManager;
	@Autowired
	private CodeManager                        codeManager;
	@Autowired
	private IEconomyManager                    economyManager;
	@Autowired
	private IEventManager                      eventManager;
	@Autowired
	private IItemManager                       itemManager;
	@Autowired
	private ILogManager                        logManager;
	@Autowired
	private IMailManager                       mailManager;
	@Autowired
	private IPaymentManager                    paymentManager;
	@Autowired
	private IPermissionManager<UUID>           permissionManager;
	@Autowired
	private IRegistrationManager               registrationManager;
	@Autowired
	@Qualifier("defaultSessionManager")
	private ISessionManager<UUID>              sessionManager;

	private boolean                            initialized = false;
	private boolean                            initRequest = false;
	
	@Autowired
	public Core()
	{
		
	}
	
	public boolean Initialize(ConfigurableApplicationContext context)
	{
		if (initialized)
		{
			return false;
		}
		
		this.context = context;

		String initRequest = context.getEnvironment().getProperty("server.init");
		if (initRequest != null)
		{
			this.initRequest = Boolean.parseBoolean(initRequest);

			if (this.initRequest)
			{
				ExternalConfigUtils.UpdateInit();
			}
		}

		version = context.getEnvironment().getProperty("version");
		serverAddress = context.getEnvironment().getProperty("server.address");
		serverAddress = context.getEnvironment().getProperty("server.port");
		serverWWWLink = context.getEnvironment().getProperty("server.wwwLink");
		
		accountManager.Initialize();
		codeManager.Initialize();
		economyManager.Initialize();
		eventManager.Initialize();
		itemManager.Initialize();
		logManager.Initialize();
		mailManager.Initialize();
		paymentManager.Initialize();
		permissionManager.Initialize();
		registrationManager.Initialize();
		sessionManager.Initialize();

		initialized = true;
		return true;
	}

	public boolean Dispose()
	{
		return true;
	}

	public <TID> LogEntry<TID> Post(LogLevel logLevel, IEntity<TID> causer, String title, String desc)
	{
		return logManager.Post(logLevel, causer, title, desc);
	}

	public LogEntry<Void> Post(LogLevel logLevel, String causer, String title, String desc)
	{
		return logManager.Post(logLevel, causer, title, desc);
	}
	
	public <TID> LogEntry<TID> Info(IEntity<TID> causer, String title, String description)
	{
		return Post(LogLevel.INFO, causer, title, description);
	}
	
	public <TID> LogEntry<Void> Info(String causer, String title, String description)
	{
		return Post(LogLevel.INFO, causer, title, description);
	}
	
	public <TID> LogEntry<TID> Warn(IEntity<TID> causer, String title, String description)
	{
		return Post(LogLevel.WARN, causer, title, description);
	}
	
	public <TID> LogEntry<Void> Warn(String causer, String title, String description)
	{
		return Post(LogLevel.WARN, causer, title, description);
	}
	
	public <TID> LogEntry<TID> Error(IEntity<TID> causer, String title, String description)
	{
		return Post(LogLevel.ERROR, causer, title, description);
	}
	
	public <TID> LogEntry<Void> Error(String causer, String title, String description)
	{
		return Post(LogLevel.ERROR, causer, title, description);
	}
	
	

	public boolean IsInitRequested()
	{
		return initRequest;
	}

	public String GetVersion()
	{
		return version;
	}

	public String GetServerAddress()
	{
		return serverAddress;
	}

	public short GetServerPort()
	{
		return serverPort;
	}

	public String GetServerWWWLink()
	{
		return serverWWWLink;
	}

	public DbCred GetDbCred()
	{
		return dbCred;
	}

	public OutputStream GetStdout()
	{
		return stdout;
	}

	public ConfigurableApplicationContext GetContext()
	{
		return context;
	}

	public Environment GetEnvironment()
	{
		return GetContext().getEnvironment();
	}

	public String GetProperty(String key)
	{
		return GetEnvironment().getProperty(key);
	}

	public EntityManagerFactory GetEntityManagerFactory()
	{
		return entityManagerFactory;
	}

	public <T> T Manager(Class<T> managerClass)
	{
		List<IManager> managers = new ArrayList<>();

		for(IManager manager:managers)
		{
			if (managerClass.isInstance(managerClass))
			{
				try
				{
					return managerClass.cast(manager);
				}
				catch(ClassCastException ex)
				{
					ex.printStackTrace();

					break;
				}
			}
		}

		return null;
	}

	public EntityManager CreateEntityManager()
	{
		return GetEntityManagerFactory().createEntityManager();
	}

	public IAccountManager GetAccountManager()
	{
		return accountManager;
	}

	public CodeManager GetCodeManager()
	{
		return codeManager;
	}

	public IEconomyManager GetEconomyManager()
	{
		return economyManager;
	}

	public IEventManager GetEventManager()
	{
		return eventManager;
	}

	public IItemManager GetItemManager()
	{
		return itemManager;
	}

	public ILogManager GetLogManager()
	{
		return logManager;
	}

	public IMailManager GetMailManager()
	{
		return mailManager;
	}

	public IPaymentManager GetPaymentManager()
	{
		return paymentManager;
	}

	public IPermissionManager<UUID> GetPermissionManager()
	{
		return permissionManager;
	}

	public IRegistrationManager GetRegistrationManager()
	{
		return registrationManager;
	}

	public ISessionManager<UUID> GetSessionManager()
	{
		return sessionManager;
	}
}
